<?php
// Start session and include files
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

include('header.php');
include('functions.php');

if (!defined('CURRENCY')) {
    define('CURRENCY', 'KES ');
}

// Initialize variables
$product_id = $_GET['id'] ?? 0;
$product_name = '';
$product_desc = '';
$product_price = '';
$error = '';
$success = '';

// Database connection
function getDBConnection() {
    $mysqli = new mysqli(DATABASE_HOST, DATABASE_USER, DATABASE_PASS, DATABASE_NAME);
    if ($mysqli->connect_error) {
        throw new Exception('Database Connection Error: ' . $mysqli->connect_error);
    }
    return $mysqli;
}

// Fetch product data
function fetchProduct($id) {
    $mysqli = getDBConnection();
    $stmt = $mysqli->prepare("SELECT product_id, product_name, product_desc, product_price FROM products WHERE product_id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $product = $result->fetch_assoc();
    $stmt->close();
    $mysqli->close();
    return $product;
}

// Update product data
function updateProduct($id, $name, $desc, $price) {
    $mysqli = getDBConnection();
    $stmt = $mysqli->prepare("UPDATE products SET product_name = ?, product_desc = ?, product_price = ? WHERE product_id = ?");
    $stmt->bind_param("ssdi", $name, $desc, $price, $id);
    $result = $stmt->execute();
    $stmt->close();
    $mysqli->close();
    return $result;
}

// Main logic
try {
    // Validate product ID
    if (!$product_id || $product_id <= 0) {
        throw new Exception('Invalid product ID!');
    }

    // Fetch existing product data
    $product = fetchProduct($product_id);
    if (!$product) {
        throw new Exception('Product not found!');
    }

    // Populate form fields
    $product_name = $product['product_name'];
    $product_desc = $product['product_desc'];
    $product_price = $product['product_price'];

    // Handle form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $action = $_POST['action'] ?? '';
        
        if ($action === 'update_product') {
            // Get form data
            $product_name = trim($_POST['product_name'] ?? '');
            $product_desc = trim($_POST['product_desc'] ?? '');
            $product_price = trim($_POST['product_price'] ?? '');
            
            // Validate form data
            if (empty($product_name) || empty($product_desc) || empty($product_price)) {
                throw new Exception('Please fill all required fields.');
            }
            
            if (!is_numeric($product_price) || $product_price < 0) {
                throw new Exception('Please enter a valid price.');
            }
            
            // Update product
            if (updateProduct($product_id, $product_name, $product_desc, $product_price)) {
                $_SESSION['success'] = 'Product updated successfully!';
                header('Location: invoice_list.php');
                exit();
            } else {
                throw new Exception('Failed to update product. Please try again.');
            }
        }
    }
} catch (Exception $e) {
    $error = $e->getMessage();
    
    // Redirect on specific errors
    if (strpos($error, 'Invalid product ID') !== false || strpos($error, 'Product not found') !== false) {
        $_SESSION['error'] = $error;
        header('Location: inoice_list.php');
        exit();
    }
}

// Display session messages
if (isset($_SESSION['error'])) {
    $error = $_SESSION['error'];
    unset($_SESSION['error']);
}

if (isset($_SESSION['success'])) {
    $success = $_SESSION['success'];
    unset($_SESSION['success']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Product - YAKUTEK</title>
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #2d3436;
            line-height: 1.6;
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
        }

        /* Card Styling */
        .card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
            margin-bottom: 20px;
        }

        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 30px;
            text-align: center;
            color: white;
        }

        .company-name {
            font-size: 28px;
            font-weight: 800;
            margin-bottom: 5px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .company-address {
            font-size: 14px;
            opacity: 0.9;
            font-weight: 500;
        }

        .page-title {
            margin-top: 15px;
        }

        .page-title h1 {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 8px;
        }

        .product-id {
            font-size: 14px;
            background: rgba(255,255,255,0.2);
            padding: 5px 15px;
            border-radius: 20px;
            display: inline-block;
        }

        .card-body {
            padding: 40px;
        }

        /* Alert Messages */
        .alert {
            padding: 15px 20px;
            margin-bottom: 25px;
            border-radius: 10px;
            font-weight: 500;
            border-left: 5px solid;
            animation: slideIn 0.3s ease;
        }

        @keyframes slideIn {
            from { transform: translateY(-10px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }

        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border-left-color: #155724;
        }

        .alert-error {
            background-color: #f8d7da;
            color: #721c24;
            border-left-color: #721c24;
        }

        .alert .close {
            float: right;
            font-size: 18px;
            font-weight: bold;
            cursor: pointer;
            opacity: 0.7;
            transition: opacity 0.3s;
        }

        .alert .close:hover {
            opacity: 1;
        }

        /* Form Styles */
        .form-group {
            margin-bottom: 25px;
        }

        label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #2d3436;
            font-size: 14px;
        }

        .form-control {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: #f8f9fa;
            font-family: 'Inter', sans-serif;
        }

        .form-control:focus {
            border-color: #667eea;
            background: white;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            outline: none;
            transform: translateY(-2px);
        }

        .input-group {
            display: flex;
            align-items: center;
        }

        .input-group-addon {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: 2px solid #667eea;
            border-right: none;
            padding: 15px 20px;
            border-radius: 10px 0 0 10px;
            font-weight: 600;
            min-width: 80px;
            text-align: center;
        }

        .input-group .form-control {
            border-radius: 0 10px 10px 0;
            border-left: none;
        }

        /* Button Styles */
        .btn-group {
            display: flex;
            gap: 15px;
            justify-content: flex-end;
            margin-top: 40px;
            padding-top: 30px;
            border-top: 2px solid #e9ecef;
        }

        .btn {
            padding: 15px 35px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            font-family: 'Inter', sans-serif;
        }

        .btn-success {
            background: linear-gradient(135deg, #00b894, #00a085);
            color: white;
            box-shadow: 0 5px 15px rgba(0, 184, 148, 0.3);
        }

        .btn-success:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(0, 184, 148, 0.4);
        }

        .btn-default {
            background: linear-gradient(135deg, #6c757d, #5a6268);
            color: white;
            box-shadow: 0 5px 15px rgba(108, 117, 125, 0.3);
        }

        .btn-default:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(108, 117, 125, 0.4);
        }

        /* Footer */
        .footer {
            text-align: center;
            color: white;
            margin-top: 30px;
            font-size: 14px;
            opacity: 0.9;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .card-body {
                padding: 25px;
            }
            
            .btn-group {
                flex-direction: column;
            }
            
            .btn {
                width: 100%;
            }
            
            .card-header {
                padding: 20px;
            }
            
            .company-name {
                font-size: 22px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Main Card -->
        <div class="card">
            <!-- Header -->
            <div class="card-header">
                <div class="company-info">
                    <div class="company-name">YAKUTEK TECHNOLOGIES LIMITED</div>
                    <div class="company-address">P.O.Box 22111 - 00400, Nairobi, Kenya</div>
                </div>
                <div class="page-title">
                    <h1>EDIT PRODUCT</h1>
                    <div class="product-id">Product ID: <?php echo htmlspecialchars($product_id); ?></div>
                </div>
            </div>

            <!-- Body -->
            <div class="card-body">
                <!-- Display Messages -->
                <?php if ($error): ?>
                    <div class='alert alert-error'>
                        <span class='close' onclick='this.parentElement.style.display="none"'>&times;</span>
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <?php if ($success): ?>
                    <div class='alert alert-success'>
                        <span class='close' onclick='this.parentElement.style.display="none"'>&times;</span>
                        <?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>

                <!-- Product Form -->
                <form method="post" id="update_product">
                    <input type="hidden" name="action" value="update_product">
                    
                    <div class="form-group">
                        <label for="product_name">Product Name *</label>
                        <input type="text" class="form-control" id="product_name" name="product_name" 
                               placeholder="Enter product name" value="<?php echo htmlspecialchars($product_name); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="product_desc">Product Description *</label>
                        <input type="text" class="form-control" id="product_desc" name="product_desc" 
                               placeholder="Enter product description" value="<?php echo htmlspecialchars($product_desc); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="product_price">Product Price *</label>
                        <div class="input-group">
                            <span class="input-group-addon"><?php echo CURRENCY; ?></span>
                            <input type="number" class="form-control" id="product_price" name="product_price" 
                                   placeholder="0.00" step="0.01" min="0" 
                                   value="<?php echo htmlspecialchars($product_price); ?>" required>
                        </div>
                    </div>
                    
                    <div class="btn-group">
                        <a href="invoice_list.php" class="btn btn-default">
                            ← Cancel
                        </a>
                        <button type="submit" class="btn btn-success">
                            ✓ Update Product
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer">
            <p>YAKUTEK TECHNOLOGIES LIMITED | +254114524239 | yakutek@musyokinet.co.ke</p>
        </div>
    </div>

    <script>
        // Enhanced form validation
        document.getElementById('update_product').addEventListener('submit', function(e) {
            const productName = document.getElementById('product_name').value.trim();
            const productDesc = document.getElementById('product_desc').value.trim();
            const productPrice = document.getElementById('product_price').value;
            
            let isValid = true;
            let errorMessage = '';
            
            // Clear previous error styles
            document.querySelectorAll('.form-control').forEach(input => {
                input.style.borderColor = '#e9ecef';
            });
            
            // Validate fields
            if (!productName) {
                document.getElementById('product_name').style.borderColor = '#dc3545';
                isValid = false;
                errorMessage = 'Product name is required.';
            } else if (!productDesc) {
                document.getElementById('product_desc').style.borderColor = '#dc3545';
                isValid = false;
                errorMessage = 'Product description is required.';
            } else if (!productPrice || parseFloat(productPrice) < 0) {
                document.getElementById('product_price').style.borderColor = '#dc3545';
                isValid = false;
                errorMessage = 'Please enter a valid price.';
            }
            
            if (!isValid) {
                e.preventDefault();
                alert('Please fix the following error:\n' + errorMessage);
            }
        });

        // Auto-dismiss alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                alert.style.display = 'none';
            });
        }, 5000);

        // Add input validation on blur
        document.querySelectorAll('.form-control').forEach(input => {
            input.addEventListener('blur', function() {
                if (this.value.trim() === '') {
                    this.style.borderColor = '#dc3545';
                } else {
                    this.style.borderColor = '#00b894';
                }
            });
            
            input.addEventListener('focus', function() {
                this.style.borderColor = '#667eea';
            });
        });
    </script>
</body>
</html>

<?php
include('footer.php');
?>