<?php
// =====================================
// Yakutech Solutions Limited Invoice Generator (Invoices Table Only)
// =====================================
// Clear any previous output buffers
while (ob_get_level()) {
    ob_end_clean();
}

// Prevent any output
ob_start();

require 'dompdf/autoload.inc.php';
include 'includes/config.php';
include 'functions.php';

use Dompdf\Dompdf;
use Dompdf\Options;

// ✅ Get invoice number
$invoice_number = $_GET['invoice_number'] ?? ($_GET['id'] ?? '');
if (empty($invoice_number)) {
    die("No invoice number provided.");
}

// ✅ Fetch invoice
$sql = "SELECT * FROM invoices WHERE invoice_number = ?";
$stmt = $mysqli->prepare($sql);
if (!$stmt) {
    die("SQL error: " . $mysqli->error);
}
$stmt->bind_param("s", $invoice_number);
$stmt->execute();
$result = $stmt->get_result();
$invoice = $result->fetch_assoc();

if (!$invoice) {
    die("Invoice not found.");
}

// ✅ Fetch invoice items
$invoice_id = $invoice['id'];
$sql_items = "SELECT * FROM invoice_items WHERE invoice_id = ?";
$stmt_items = $mysqli->prepare($sql_items);
if (!$stmt_items) {
    die("SQL error: " . $mysqli->error);
}
$stmt_items->bind_param("i", $invoice_id);
$stmt_items->execute();
$items_result = $stmt_items->get_result();

// ✅ Totals
$subtotal = 0;
$vat_rate = 0.16;
while ($row = $items_result->fetch_assoc()) {
    $subtotal += $row['amount'];
}
$vat_amount = $subtotal * $vat_rate;
$grand_total = $subtotal + $vat_amount;
$items_result->data_seek(0);

// ✅ Generate random CU Invoice Number (17 digits)
function generateCUInvoiceNumber() {
    $random_digits = '';
    for ($i = 0; $i < 13; $i++) {
        $random_digits .= mt_rand(0, 9);
    }
    return '0170' . $random_digits;
}

$cu_invoice_no = generateCUInvoiceNumber();

// ✅ FIX FOR CPANEL: Logo handling
$base64_logo = '';
$logo_found = false;

// Common cPanel logo paths - try each one
$possible_logo_paths = [
    // If your system is in main directory
    $_SERVER['DOCUMENT_ROOT'] . '/images/logo.png',
    $_SERVER['DOCUMENT_ROOT'] . '/Invoice-System-In-PHP-main/images/logo.png',
    
    // If your system is in subdirectory
    dirname(__FILE__) . '/images/logo.png',
    dirname(__FILE__) . '/../images/logo.png',
    
    // Absolute paths (common cPanel structure)
    '/home/' . (isset($_SERVER['CPANEL']) ? get_current_user() : 'username') . '/public_html/images/logo.png',
    '/home/' . (isset($_SERVER['CPANEL']) ? get_current_user() : 'username') . '/public_html/Invoice-System-In-PHP-main/images/logo.png',
];

foreach ($possible_logo_paths as $logo_path) {
    if (file_exists($logo_path) && is_readable($logo_path)) {
        try {
            $logo_info = getimagesize($logo_path);
            $logo_data = file_get_contents($logo_path);
            if ($logo_data !== false) {
                $base64_logo = 'data:' . $logo_info['mime'] . ';base64,' . base64_encode($logo_data);
                $logo_found = true;
                break;
            }
        } catch (Exception $e) {
            // Continue to next path if this one fails
            continue;
        }
    }
}

// If still not found, try to create a simple SVG logo as fallback
if (!$logo_found) {
    // Create a simple SVG logo as base64
    $svg_logo = '<svg width="90" height="90" xmlns="http://www.w3.org/2000/svg">
        <circle cx="45" cy="45" r="40" fill="#6a0dad" />
        <text x="45" y="35" text-anchor="middle" fill="white" font-family="Arial" font-size="12" font-weight="bold">YAKUTECH</text>
        <text x="45" y="50" text-anchor="middle" fill="white" font-family="Arial" font-size="8">SOLUTIONS</text>
        <text x="45" y="60" text-anchor="middle" fill="white" font-family="Arial" font-size="8">LIMITED</text>
    </svg>';
    $base64_logo = 'data:image/svg+xml;base64,' . base64_encode($svg_logo);
    $logo_found = true;
}

// ✅ Build HTML
$html = '
<html>
<head>
    <style>
        @page { margin: 1cm; }
        body { font-family: "DejaVu Sans", Arial, sans-serif; font-size: 10px; color: #000; }
        .invoice-wrapper { width: 100%; max-width: 21cm; margin: auto; }
        .header { text-align: center; border-bottom: 2px solid #6a0dad; padding-bottom: 10px; margin-bottom: 15px; }
        .header img { width: 90px; height: 90px; border-radius: 50%; border: 2px solid #6a0dad; object-fit: cover; margin-bottom: 5px; }
        .company-name { font-size: 16px; color: #6a0dad; font-weight: bold; }
        .company-details { font-size: 9px; line-height: 1.2; }
        .invoice-title { text-align: center; font-size: 14px; font-weight: bold; color: #6a0dad; margin: 10px 0; }
        .section { margin-bottom: 10px; border: 1px solid #ddd; padding: 8px; font-size: 9px; }
        .section-title { font-weight: bold; margin-bottom: 4px; color: #6a0dad; }
        table { width: 100%; border-collapse: collapse; margin-top: 8px; }
        th, td { border: 1px solid #000; padding: 4px 6px; font-size: 8px; }
        th { background-color: #f4e6ff; font-weight: bold; }
        .text-right { text-align: right; }
        .text-center { text-align: center; }
        .bank-section { border: 1px solid #ddd; padding: 6px; font-size: 9px; margin: 10px 0; }
        .totals { width: 100%; border-collapse: collapse; margin-top: 8px; font-size: 9px; }
        .totals td { padding: 5px; border: 1px solid #000; }
        .footer { text-align: center; margin-top: 15px; font-size: 9px; color: #6a0dad; }
        .footer strong { color: #000; }
    </style>
</head>
<body>
    <div class="invoice-wrapper">
        <div class="header">';

// ✅ Logo display - Always use base64 for cPanel reliability
$html .= '<img src="' . $base64_logo . '" alt="Yakutech Logo">';

$html .= '
            <div class="company-name">YAKUTECH SOLUTIONS LIMITED</div>
            <div class="company-details">
                P.O.Box Nairobi 00100, Kenya<br>
                PIN: Y01234567899 | Email: yakutech@gmail.com | Phone: 0114524239
            </div>
        </div>

        <div class="invoice-title">INVOICE</div>

        <div class="section">
            <div class="section-title">Invoice To:</div>
            <div>
                <strong>' . htmlspecialchars($invoice['customer_name'] ?? '') . '</strong><br>
                ' . nl2br(htmlspecialchars($invoice['customer_address'] ?? '')) . '<br>
                ' . htmlspecialchars($invoice['customer_city'] ?? '') . ', ' . htmlspecialchars($invoice['customer_country'] ?? '') . '<br>
                ' . (!empty($invoice['customer_postal']) ? 'P.O.Box ' . htmlspecialchars($invoice['customer_postal']) . '<br>' : '') . '
                ' . (!empty($invoice['customer_email']) ? 'Email: ' . htmlspecialchars($invoice['customer_email']) . '<br>' : '') . '
                ' . (!empty($invoice['customer_phone']) ? 'Phone: ' . htmlspecialchars($invoice['customer_phone']) . '<br>' : '') . '
                ' . (!empty($invoice['customer_pin']) ? 'PIN: ' . htmlspecialchars($invoice['customer_pin']) : '') . '
            </div>
        </div>

        <table>
            <tr>
                <th>LPO No.</th>
                <th>Terms</th>
                <th>Customer PIN</th>
                <th>CU Invoice No.</th>
                <th>Yakutech PIN</th>
                <th>Date</th>
                <th>Invoice No.</th>
            </tr>
            <tr>
                <td>' . ($invoice['lpo_number'] ?? '') . '</td>
                <td>' . ($invoice['terms'] ?? '') . '</td>
                <td>' . ($invoice['customer_pin'] ?? '') . '</td>
                <td>' . $cu_invoice_no . '</td>
                <td><strong>Y01234567899</strong></td>
                <td>' . ($invoice['invoice_date'] ?? date("d-m-Y")) . '</td>
                <td><strong>' . ($invoice['invoice_number'] ?? 'INV0001') . '</strong></td>
            </tr>
        </table>

        <table>
            <thead>
                <tr>
                    <th>Item</th>
                    <th>Description</th>
                    <th>Qty</th>
                    <th>Rate</th>
                    <th>Amount (KES)</th>
                </tr>
            </thead>
            <tbody>';

if ($items_result->num_rows > 0) {
    while ($row = $items_result->fetch_assoc()) {
        $unit_price = ($row['quantity'] > 0) ? $row['amount'] / $row['quantity'] : 0;
        $html .= '
            <tr>
                <td>' . htmlspecialchars($row['item_code']) . '</td>
                <td>' . htmlspecialchars($row['description']) . '</td>
                <td class="text-center">' . $row['quantity'] . '</td>
                <td class="text-right">' . number_format($unit_price, 2) . '</td>
                <td class="text-right">' . number_format($row['amount'], 2) . '</td>
            </tr>';
    }
}

$html .= '
            </tbody>
        </table>

        <div class="bank-section">
            <strong>Bank Details:</strong><br>
            KCB Bank, Kayamba Branch<br>
            A/C: 00000000000001
        </div>

        <table class="totals">
            <tr><td><strong>Sub-Total</strong></td><td class="text-right">KES ' . number_format($subtotal, 2) . '</td></tr>
            <tr><td><strong>VAT (16%)</strong></td><td class="text-right">KES ' . number_format($vat_amount, 2) . '</td></tr>
            <tr><td><strong>Grand Total</strong></td><td class="text-right"><strong>KES ' . number_format($grand_total, 2) . '</strong></td></tr>
        </table>

        <div class="footer"><strong>THANK YOU</strong> FOR CHOOSING YAKUTECH SOLUTIONS LIMITED</div>
    </div>
</body>
</html>';

// ✅ Generate PDF with cPanel optimized settings
$options = new Options();
$options->set('isRemoteEnabled', true);
$options->set('isHtml5ParserEnabled', true);
$options->set('defaultFont', 'DejaVu Sans');
$options->set('chroot', $_SERVER['DOCUMENT_ROOT']);
$options->set('tempDir', sys_get_temp_dir()); // Use system temp directory
$options->set('logOutputFile', sys_get_temp_dir() . '/dompdf_log.txt');

$dompdf = new Dompdf($options);
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');
$dompdf->render();

// Output the PDF
$dompdf->stream("invoice_" . ($invoice['invoice_number'] ?? 'INV0001') . ".pdf", ["Attachment" => false]);
exit;
?>