<?php
// Include database config and functions
include_once('includes/config.php');
include('functions.php');

// Connect to database
$mysqli = new mysqli(DATABASE_HOST, DATABASE_USER, DATABASE_PASS, DATABASE_NAME);
if ($mysqli->connect_error) {
    die('Database connection failed: ' . $mysqli->connect_error);
}

// Errors and success messages
$errors = [];

// Function to auto-generate invoice number
function generateInvoiceNumber($mysqli) {
    $year = date('Y');
    $month = date('m');

    $query = "SELECT invoice_number FROM invoices WHERE invoice_number LIKE 'INV-$year-$month-%' ORDER BY id DESC LIMIT 1";
    $result = $mysqli->query($query);

    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $last_number = explode('-', $row['invoice_number']);
        $next_seq = intval(end($last_number)) + 1;
        return "INV-$year-$month-" . str_pad($next_seq, 4, '0', STR_PAD_LEFT);
    } else {
        return "INV-$year-$month-0001";
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create_invoice') {

    $invoice_number = generateInvoiceNumber($mysqli);
    $invoice_date   = $_POST['invoice_date'] ?? date('Y-m-d');
    $invoice_type   = $_POST['invoice_type'] ?? 'invoice';
    $invoice_status = $_POST['invoice_status'] ?? 'open';
    $lpo_number     = $_POST['lpo_number'] ?? '';
    $terms          = $_POST['terms'] ?? '';
    $customer_pin   = $_POST['customer_pin'] ?? '';
    $company_pin    = 'P051683953Q';
    $notes          = $_POST['notes'] ?? '';

    $customer_name    = $_POST['customer_name'] ?? '';
    $customer_email   = $_POST['customer_email'] ?? '';
    $customer_address = $_POST['customer_address'] ?? '';
    $customer_city    = $_POST['customer_city'] ?? '';
    $customer_postal  = $_POST['customer_postal'] ?? '';
    $customer_country = $_POST['customer_country'] ?? '';

    $items = $_POST['items'] ?? [];
    $discount_percentage = floatval($_POST['discount_percentage'] ?? 0);
    $vat_rate = floatval($_POST['vat_rate'] ?? 16);

    // Validation
    if (empty($customer_name)) $errors[] = 'Customer name is required';
    if (empty($items)) $errors[] = 'At least one item is required';

    $valid_items = 0;
    foreach ($items as $item) {
        if (!empty($item['description']) && floatval($item['qty']) > 0 && floatval($item['rate']) > 0) {
            $valid_items++;
        }
    }
    if ($valid_items === 0) $errors[] = 'At least one valid item is required';

    // Insert invoice if no errors
    if (empty($errors)) {
        $subtotal = 0;
        foreach ($items as $item) {
            if (!empty($item['description']) && floatval($item['qty']) > 0 && floatval($item['rate']) > 0) {
                $subtotal += floatval($item['qty']) * floatval($item['rate']);
            }
        }

        $discount_amount = ($subtotal * $discount_percentage) / 100;
        $after_discount  = $subtotal - $discount_amount;
        $vat_amount      = ($after_discount * $vat_rate) / 100;
        $total           = $after_discount + $vat_amount;

        $current_user_id = $_SESSION['user_id'] ?? 1;

      $stmt = $mysqli->prepare("INSERT INTO invoices 
    (invoice_number, invoice_date, invoice_type, invoice_status, lpo_number, terms, 
    customer_pin, company_pin, customer_name, customer_email, customer_address, customer_city, 
    customer_postal, customer_country, subtotal, discount_percentage, discount_amount, 
    vat_rate, vat_amount, total, notes, created_by, date_created)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");


        if ($stmt) {
           $stmt->bind_param(
    "ssssssssssssssddddddsi",
    $invoice_number, $invoice_date, $invoice_type, $invoice_status, $lpo_number, $terms,
    $customer_pin, $company_pin, $customer_name, $customer_email, $customer_address,
    $customer_city, $customer_postal, $customer_country,
    $subtotal, $discount_percentage, $discount_amount,
    $vat_rate, $vat_amount, $total, $notes, $current_user_id
);


            if ($stmt->execute()) {
                $invoice_id = $stmt->insert_id;

                // Insert invoice items
                $item_stmt = $mysqli->prepare("INSERT INTO invoice_items 
                    (invoice_id, item_code, description, quantity, rate, amount)
                    VALUES (?, ?, ?, ?, ?, ?)");

                if ($item_stmt) {
                    foreach ($items as $item) {
                        if (!empty($item['description']) && floatval($item['qty']) > 0 && floatval($item['rate']) > 0) {
                            $code = $item['code'] ?? '';
                            $desc = $item['description'];
                            $qty  = floatval($item['qty']);
                            $rate = floatval($item['rate']);
                            $amt  = $qty * $rate;

                            $item_stmt->bind_param("issddd", $invoice_id, $code, $desc, $qty, $rate, $amt);
                            $item_stmt->execute();
                        }
                    }
                    $item_stmt->close();
                }

                // ✅ Flash success message & redirect
                $_SESSION['success_message'] = "Invoice #$invoice_number created successfully!";
                echo '<script>window.location.href = "invoice-list.php";</script>';
                exit;
            } else {
                $errors[] = "Database error: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $errors[] = "Prepare failed: " . $mysqli->error;
        }
    }
}

include('header.php');
?>

<div class="row">
    <div class="col-md-12">
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <h5><i class="fas fa-exclamation-triangle"></i> Please fix the following errors:</h5>
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        <?php if (!empty($success)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Company Header -->
<div class="company-header">
    <div class="company-logo">
        <i class="fas fa-file-invoice-dollar"></i>
    </div>
    <div class="company-info">
        <h1>YAKUTECH SOLUTIONS LIMITED</h1>
        <p class="company-address">P.O.Box 22111 - 00400, Nairobi, Kenya</p>
        <p class="company-contact">
            <i class="fas fa-phone"></i> +254114524239 | 
            <i class="fas fa-envelope"></i> yakutech@gmail.com
        </p>
    </div>
    <div class="document-type-badge">
        <div class="badge invoice-badge">
            <i class="fas fa-receipt"></i>
            <span>INVOICE</span>
        </div>
    </div>
</div>

<div class="invoice-creator">
    <div class="creator-header">
        <div class="header-icon">
            <i class="fas fa-plus-circle"></i>
        </div>
        <div class="header-title">
            <h2>Create New Invoice</h2>
            <p>Fill in the details below to create a new invoice</p>
        </div>
        <div class="bill-to-section">
            <div class="bill-to-header">
                <i class="fas fa-user"></i>
                <span>BILL TO:</span>
            </div>
        </div>
    </div>

    <div class="creator-body">
        <form method="post" id="create_invoice" class="invoice-form">
            <input type="hidden" name="action" value="create_invoice">

            <!-- Invoice & Customer Details -->
            <div class="form-section">
                <div class="section-header">
                    <i class="fas fa-file-alt"></i>
                    <h3>Invoice Details</h3>
                </div>
                <div class="section-content">
                    <div class="row g-4">
                        <div class="col-md-6">
                            <div class="detail-card">
                                <h4 class="card-title">Invoice Information</h4>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Invoice Number</label>
                                            <input type="text" class="form-control" value="<?php echo generateInvoiceNumber($mysqli); ?>" readonly>
                                            <div class="form-text">Auto-generated</div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Date</label>
                                            <input type="date" class="form-control" name="invoice_date" value="<?php echo date('Y-m-d'); ?>" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Invoice Type</label>
                                            <select name="invoice_type" class="form-select">
                                                <option value="invoice" selected>Invoice</option>
                                                <option value="quote">Quote</option>
                                                <option value="receipt">Receipt</option>
                                                <option value="proforma">Proforma</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Status</label>
                                            <select name="invoice_status" class="form-select">
                                                <option value="open" selected>Open</option>
                                                <option value="paid">Paid</option>
                                                <option value="pending">Pending</option>
                                                <option value="cancelled">Cancelled</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-12">
                                        <div class="form-group">
                                            <label class="form-label">LPO Number</label>
                                            <input type="text" class="form-control" name="lpo_number" placeholder="Enter LPO number">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="detail-card">
                                <h4 class="card-title">Customer Information</h4>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label required">Customer Name</label>
                                            <input type="text" class="form-control" name="customer_name" placeholder="Enter customer name" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Customer Email</label>
                                            <input type="email" class="form-control" name="customer_email" placeholder="customer@example.com">
                                            <div class="form-text">Required for email invoices</div>
                                        </div>
                                    </div>
                                    <div class="col-12">
                                        <div class="form-group">
                                            <label class="form-label">Address</label>
                                            <textarea class="form-control" name="customer_address" rows="2" placeholder="Customer address"></textarea>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label class="form-label">City</label>
                                            <input type="text" class="form-control" name="customer_city" placeholder="City">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label class="form-label">Postal Code</label>
                                            <input type="text" class="form-control" name="customer_postal" placeholder="Postal code">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label class="form-label">Country</label>
                                            <input type="text" class="form-control" name="customer_country" value="Kenya">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Customer PIN</label>
                                            <input type="text" class="form-control" name="customer_pin" placeholder="Customer PIN">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label class="form-label">Company PIN</label>
                                            <input type="text" class="form-control" value="P051683953Q" readonly>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Items Table -->
            <div class="form-section">
                <div class="section-header">
                    <i class="fas fa-list-ol"></i>
                    <h3>Items & Services</h3>
                </div>
                <div class="section-content">
                    <div class="items-table-container">
                        <table class="items-table" id="items-table">
                            <thead>
                                <tr>
                                    <th width="12%">Item Code</th>
                                    <th width="38%">Description</th>
                                    <th width="10%">Qty</th>
                                    <th width="15%">Rate (KES)</th>
                                    <th width="15%">Amount (KES)</th>
                                    <th width="10%">Action</th>
                                </tr>
                            </thead>
                            <tbody id="items-tbody">
                                <tr class="item-row">
                                    <td><input type="text" class="form-control" name="items[0][code]" placeholder="Code"></td>
                                    <td><input type="text" class="form-control" name="items[0][description]" placeholder="Item description" required></td>
                                    <td><input type="number" class="form-control qty-input" name="items[0][qty]" step="0.01" value="1" min="0" required></td>
                                    <td><input type="number" class="form-control rate-input" name="items[0][rate]" step="0.01" min="0" placeholder="0.00" required></td>
                                    <td><input type="text" class="form-control amount-input" readonly></td>
                                    <td class="text-center">
                                        <button type="button" class="btn btn-danger btn-sm remove-item" title="Remove item">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                        <div class="table-actions">
                            <button type="button" class="btn btn-success" id="add-item">
                                <i class="fas fa-plus"></i> Add Item
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Totals & Notes -->
            <div class="form-section">
                <div class="section-header">
                    <i class="fas fa-calculator"></i>
                    <h3>Totals & Notes</h3>
                </div>
                <div class="section-content">
                    <div class="row g-4">
                        <div class="col-md-6">
                            <div class="notes-card">
                                <h4 class="card-title">Additional Information</h4>
                                <div class="form-group">
                                    <label class="form-label">Payment Terms</label>
                                    <input type="text" class="form-control" name="terms" placeholder="e.g., Net 30 days">
                                </div>
                                <div class="form-group">
                                    <label class="form-label">Notes</label>
                                    <textarea class="form-control" name="notes" rows="4" placeholder="Additional notes or comments..."></textarea>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="totals-card">
                                <h4 class="card-title">Invoice Summary</h4>
                                <div class="summary-table">
                                    <div class="summary-row">
                                        <span>Sub-Total:</span>
                                        <span>KES <span id="subtotal">0.00</span></span>
                                    </div>
                                    <div class="summary-row">
                                        <span>
                                            Discount:
                                            <input type="number" class="form-control-sm" name="discount_percentage" id="discount_percentage" value="0" min="0" max="100">%
                                        </span>
                                        <span>KES <span id="discount_amount">0.00</span></span>
                                    </div>
                                    <div class="summary-row">
                                        <span>
                                            VAT:
                                            <input type="number" class="form-control-sm" name="vat_rate" id="vat_rate" value="16" min="0" max="100">%
                                        </span>
                                        <span>KES <span id="vat_amount">0.00</span></span>
                                    </div>
                                    <div class="summary-row total">
                                        <span>TOTAL:</span>
                                        <span>KES <span id="total">0.00</span></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Terms & Conditions -->
            <div class="form-section">
                <div class="section-header">
                    <i class="fas fa-file-contract"></i>
                    <h3>Terms & Conditions</h3>
                </div>
                <div class="section-content">
                    <div class="terms-card">
                        <ul class="terms-list">
                            <li>Overdue accounts will be charged an interest of 3% per month.</li>
                            <li>Goods once sold are not returnable.</li>
                            <li>Title of goods remains with YAKUTECH SOLUTIONS LIMITED until fully paid for.</li>
                            <li>All disputes are subject to Nairobi jurisdiction.</li>
                        </ul>
                        <div class="terms-footer">
                            <p class="thank-you">THANK YOU FOR CHOOSING YAKUTECH SOLUTIONS LIMITED</p>
                            <p class="eoe">E & O.E</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Submit Buttons -->
            <div class="form-actions">
                <button type="submit" class="btn btn-primary btn-lg">
                    <i class="fas fa-save"></i> Create Invoice
                </button>
                <button type="reset" class="btn btn-outline-secondary btn-lg">
                    <i class="fas fa-redo"></i> Reset Form
                </button>
                <a href="invoices.php" class="btn btn-outline-danger btn-lg">
                    <i class="fas fa-times"></i> Cancel
                </a>
            </div>
        </form>
    </div>
</div>

<style>
/* ===== VARIABLES ===== */
:root {
    --primary: #2E8B57;
    --primary-dark: #228B22;
    --primary-light: #3CB371;
    --secondary: #6c757d;
    --success: #28a745;
    --danger: #dc3545;
    --warning: #ffc107;
    --info: #17a2b8;
    --light: #f8f9fa;
    --dark: #343a40;
    --border: #dee2e6;
    --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
    --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
    --shadow-lg: 0 10px 15px rgba(0,0,0,0.1);
    --radius: 8px;
    --radius-sm: 4px;
}

/* ===== COMPANY HEADER ===== */
.company-header {
    background: linear-gradient(135deg, var(--primary), var(--primary-light));
    color: white;
    padding: 2rem;
    margin-bottom: 2rem;
    border-radius: var(--radius);
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 1.5rem;
    box-shadow: var(--shadow-lg);
    position: relative;
}

.company-logo {
    font-size: 3rem;
    opacity: 0.9;
}

.company-info {
    flex: 1;
}

.company-info h1 {
    margin: 0 0 0.5rem 0;
    font-size: 1.8rem;
    font-weight: 700;
}

.company-address {
    margin: 0.25rem 0;
    opacity: 0.9;
}

.company-contact {
    margin: 0.25rem 0;
    opacity: 0.9;
}

.company-contact i {
    margin-right: 0.5rem;
}

/* Document Type Badge */
.document-type-badge {
    position: absolute;
    top: 20px;
    right: 20px;
}

.badge {
    padding: 12px 20px;
    border-radius: 25px;
    font-weight: 700;
    font-size: 1rem;
    text-transform: uppercase;
    letter-spacing: 1px;
    box-shadow: var(--shadow-md);
}

.invoice-badge {
    background: linear-gradient(135deg, #FF6B6B, #FF8E53);
    color: white;
    display: flex;
    align-items: center;
    gap: 8px;
}

.invoice-badge i {
    font-size: 1.2rem;
}

/* ===== INVOICE CREATOR ===== */
.invoice-creator {
    background: white;
    border-radius: var(--radius);
    box-shadow: var(--shadow-md);
    margin-bottom: 2rem;
    overflow: hidden;
}

.creator-header {
    background: linear-gradient(135deg, var(--light), #ffffff);
    padding: 1.5rem 2rem;
    border-bottom: 3px solid var(--primary);
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 1rem;
    position: relative;
}

.header-icon {
    font-size: 2rem;
    color: var(--primary);
}

.header-title h2 {
    margin: 0;
    color: var(--dark);
    font-weight: 600;
}

.header-title p {
    margin: 0.25rem 0 0 0;
    color: var(--secondary);
}

/* Bill To Section */
.bill-to-section {
    text-align: right;
}

.bill-to-header {
    background: var(--primary);
    color: white;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    font-weight: 600;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    box-shadow: var(--shadow-sm);
}

.bill-to-header i {
    font-size: 1.1rem;
}

.creator-body {
    padding: 2rem;
}

/* ===== FORM SECTIONS ===== */
.form-section {
    margin-bottom: 2rem;
    border: 1px solid var(--border);
    border-radius: var(--radius);
    overflow: hidden;
}

.section-header {
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
    padding: 1rem 1.5rem;
    border-bottom: 1px solid var(--border);
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.section-header i {
    color: var(--primary);
    font-size: 1.2rem;
}

.section-header h3 {
    margin: 0;
    font-size: 1.2rem;
    font-weight: 600;
    color: var(--dark);
}

.section-content {
    padding: 1.5rem;
}

/* ===== CARDS ===== */
.detail-card, .notes-card, .totals-card {
    background: var(--light);
    padding: 1.5rem;
    border-radius: var(--radius);
    border-left: 4px solid var(--primary);
}

.card-title {
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 1rem;
    color: var(--dark);
    border-bottom: 2px solid var(--primary);
    padding-bottom: 0.5rem;
}

/* ===== FORM ELEMENTS ===== */
.form-group {
    margin-bottom: 1rem;
}

.form-label {
    font-weight: 600;
    color: var(--dark);
    margin-bottom: 0.5rem;
    display: block;
}

.form-label.required::after {
    content: " *";
    color: var(--danger);
}

.form-control, .form-select {
    border: 1px solid var(--border);
    border-radius: var(--radius-sm);
    padding: 0.75rem;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: var(--primary);
    box-shadow: 0 0 0 0.2rem rgba(46, 139, 87, 0.25);
}

.form-control-sm {
    width: 80px;
    display: inline-block;
    margin-left: 0.5rem;
    padding: 0.25rem 0.5rem;
    height: auto;
}

.form-text {
    font-size: 0.875rem;
    color: var(--secondary);
    margin-top: 0.25rem;
}

/* ===== ITEMS TABLE ===== */
.items-table-container {
    background: white;
    border-radius: var(--radius);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
}

.items-table {
    width: 100%;
    border-collapse: collapse;
}

.items-table th {
    background: linear-gradient(135deg, var(--primary), var(--primary-light));
    color: white;
    font-weight: 600;
    padding: 1rem 0.75rem;
    text-align: left;
    font-size: 0.9rem;
}

.items-table td {
    padding: 1rem 0.75rem;
    border-bottom: 1px solid var(--border);
    vertical-align: middle;
}

.items-table tr:hover {
    background-color: #f8f9fa;
}

.items-table input[readonly] {
    background-color: #f8f9fa;
    border-color: #e9ecef;
}

.table-actions {
    padding: 1rem;
    background: var(--light);
    text-align: right;
}

/* ===== SUMMARY TABLE ===== */
.summary-table {
    background: white;
    border-radius: var(--radius-sm);
    overflow: hidden;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem;
    border-bottom: 1px solid var(--border);
}

.summary-row:last-child {
    border-bottom: none;
}

.summary-row.total {
    background: linear-gradient(135deg, var(--primary), var(--primary-light));
    color: white;
    font-weight: 700;
    font-size: 1.1rem;
}

/* ===== TERMS & CONDITIONS ===== */
.terms-card {
    background: linear-gradient(135deg, #fffaf0, #fff);
    border: 2px solid var(--warning);
    border-radius: var(--radius);
    padding: 1.5rem;
}

.terms-list {
    margin: 0;
    padding-left: 1.5rem;
}

.terms-list li {
    margin-bottom: 0.5rem;
    color: var(--dark);
}

.terms-footer {
    text-align: center;
    margin-top: 1.5rem;
    padding-top: 1rem;
    border-top: 1px solid var(--border);
}

.thank-you {
    font-weight: 700;
    color: var(--primary);
    margin: 0;
}

.eoe {
    font-style: italic;
    color: var(--secondary);
    margin: 0.25rem 0 0 0;
}

/* ===== FORM ACTIONS ===== */
.form-actions {
    background: var(--light);
    padding: 1.5rem;
    border-radius: var(--radius);
    text-align: center;
    margin-top: 2rem;
}

.form-actions .btn {
    margin: 0 0.5rem;
    padding: 0.75rem 2rem;
    font-weight: 600;
    border-radius: var(--radius);
}

.btn-primary {
    background: linear-gradient(135deg, var(--primary), var(--primary-light));
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, var(--primary-dark), var(--primary));
    transform: translateY(-1px);
}

/* ===== ALERTS ===== */
.alert {
    border-radius: var(--radius);
    border: none;
    box-shadow: var(--shadow-sm);
}

.alert-danger {
    background: linear-gradient(135deg, #f8d7da, #f5c6cb);
    color: #721c24;
}

.alert-success {
    background: linear-gradient(135deg, #d4edda, #c3e6cb);
    color: #155724;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .company-header {
        flex-direction: column;
        text-align: center;
        padding: 1.5rem;
    }
    
    .document-type-badge {
        position: static;
        margin-top: 1rem;
    }
    
    .creator-header {
        flex-direction: column;
        text-align: center;
        gap: 0.5rem;
    }
    
    .bill-to-section {
        text-align: center;
        margin-top: 1rem;
    }
    
    .items-table {
        font-size: 0.875rem;
    }
    
    .items-table th,
    .items-table td {
        padding: 0.5rem;
    }
    
    .form-actions .btn {
        display: block;
        width: 100%;
        margin: 0.5rem 0;
    }
    
    .section-content {
        padding: 1rem;
    }
}

/* ===== ANIMATIONS ===== */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.form-section {
    animation: fadeIn 0.5s ease-out;
}

/* ===== UTILITY CLASSES ===== */
.text-center { text-align: center; }
.text-right { text-align: right; }
.mb-0 { margin-bottom: 0; }
.mt-2 { margin-top: 0.5rem; }
.g-3 { gap: 1rem; }
.g-4 { gap: 1.5rem; }

/* ===== PRINT STYLES ===== */
@media print {
    .form-actions, .alert {
        display: none;
    }
    
    .invoice-creator {
        box-shadow: none;
        border: 1px solid #000;
    }
    
    .badge {
        background: #000 !important;
        color: white !important;
    }
}
</style>

<script>
$(document).ready(function() {
    let itemCount = 1;
    
    // Add new item row
    $('#add-item').click(function() {
        const newRow = `
            <tr class="item-row">
                <td><input type="text" class="form-control" name="items[${itemCount}][code]" placeholder="Code"></td>
                <td><input type="text" class="form-control" name="items[${itemCount}][description]" placeholder="Item description" required></td>
                <td><input type="number" class="form-control qty-input" name="items[${itemCount}][qty]" step="0.01" value="1" min="0" required></td>
                <td><input type="number" class="form-control rate-input" name="items[${itemCount}][rate]" step="0.01" min="0" placeholder="0.00" required></td>
                <td><input type="text" class="form-control amount-input" readonly></td>
                <td class="text-center">
                    <button type="button" class="btn btn-danger btn-sm remove-item" title="Remove item">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>`;
        $('#items-tbody').append(newRow);
        itemCount++;
    });

    // Remove item row
    $(document).on('click', '.remove-item', function() {
        if ($('#items-tbody tr').length > 1) {
            $(this).closest('tr').remove();
            calculateTotals();
        } else {
            alert('At least one item is required');
        }
    });

    // Calculate amount when qty or rate changes
    $(document).on('input', '.qty-input, .rate-input', function() {
        const row = $(this).closest('tr');
        const qty = parseFloat(row.find('.qty-input').val()) || 0;
        const rate = parseFloat(row.find('.rate-input').val()) || 0;
        const amount = qty * rate;
        row.find('.amount-input').val(amount.toFixed(2));
        calculateTotals();
    });

    // Calculate totals when discount or VAT changes
    $(document).on('input', '#discount_percentage, #vat_rate', function() {
        calculateTotals();
    });

    // Main totals calculation function
    function calculateTotals() {
        let subtotal = 0;
        
        $('.item-row').each(function() {
            const qty = parseFloat($(this).find('.qty-input').val()) || 0;
            const rate = parseFloat($(this).find('.rate-input').val()) || 0;
            subtotal += qty * rate;
        });

        const discountPercentage = parseFloat($('#discount_percentage').val()) || 0;
        const vatRate = parseFloat($('#vat_rate').val()) || 0;
        
        const discountAmount = (subtotal * discountPercentage) / 100;
        const afterDiscount = subtotal - discountAmount;
        const vatAmount = (afterDiscount * vatRate) / 100;
        const total = afterDiscount + vatAmount;

        $('#subtotal').text(subtotal.toFixed(2));
        $('#discount_amount').text(discountAmount.toFixed(2));
        $('#vat_amount').text(vatAmount.toFixed(2));
        $('#total').text(total.toFixed(2));
    }

    // Auto-calculate on page load
    calculateTotals();

    // Form validation
    $('#create_invoice').on('submit', function(e) {
        let valid = true;
        let hasValidItems = false;

        $('.item-row').each(function() {
            const description = $(this).find('input[name*="[description]"]').val().trim();
            const qty = parseFloat($(this).find('.qty-input').val()) || 0;
            const rate = parseFloat($(this).find('.rate-input').val()) || 0;
            
            if (description && qty > 0 && rate > 0) {
                hasValidItems = true;
            }
        });

        if (!hasValidItems) {
            alert('Please add at least one valid item with description, quantity, and rate.');
            valid = false;
        }

        if (!$('input[name="customer_name"]').val().trim()) {
            alert('Customer name is required.');
            valid = false;
        }

        if (!valid) {
            e.preventDefault();
        }
    });
});
</script>

<?php include('footer.php'); ?>